import { BlockPermutation, BlockLocationIterator, EntityEquippableComponent, EquipmentSlot, GameMode, ItemStack, Player } from '@minecraft/server';
import { MinecraftBlockItemTypes } from '../item/MinecraftBlockItemTypes';
export class Helper {
  static compareVector3(v1, v2) {
    return v1.x === v2.x && v1.y === v2.y && v1.z === v2.z;
  }
  static stringifyVector3(v, includeCommas = false) {
    const c = includeCommas ? ',' : '';
    return `${v.x}${c} ${v.y}${c} ${v.z}`;
  }
  static isItemOrBlockItem(item) {
    if (MinecraftBlockItemTypes.includes(item.typeId)) {
      return true;
    }
    try {
      BlockPermutation.resolve(item.typeId);
      return false;
    } catch (_err) {
      return true;
    }
  }
  static forIterator(iterator, callback) {
    let result = iterator.next();
    while (!result.done) {
      if (callback(result.value)) break;
      result = iterator.next();
    }
  }
  static decreaseMainhandItemStack(player, amount = 1, itemStack = undefined) {
    if (player.getGameMode() === GameMode.creative) return;
    const equipments = player.getComponent(EntityEquippableComponent.componentId);
    const mainhandItem = itemStack || equipments.getEquipment(EquipmentSlot.Mainhand);
    const previousItemAmount = mainhandItem.amount;
    if (previousItemAmount > amount) mainhandItem.amount -= amount;
    equipments.setEquipment(EquipmentSlot.Mainhand, previousItemAmount > amount ? mainhandItem : null);
  }
}
export class Vector3 {
  constructor(xOrArrayOrObject, y, z) {
    if (Array.isArray(xOrArrayOrObject)) {
      [this.x, this.y, this.z] = xOrArrayOrObject;
    }
    else if (typeof xOrArrayOrObject === "object") {
      this.x = xOrArrayOrObject.x;
      this.y = xOrArrayOrObject.y;
      this.z = xOrArrayOrObject.z;
    }
    else {
      this.x = xOrArrayOrObject;
      this.y = y;
      this.z = z;
    }
  }
  static fromMinecraftVector3(v) {
    return new Vector3(v.x, v.y, v.z);
  }
  toMinecraftVector3() {
    return { x: this.x, y: this.y, z: this.z };
  }
  add(...vectors) {
    return new Vector3(this.x + vectors.reduce((sum, v) => sum + v.x, 0), this.y + vectors.reduce((sum, v) => sum + v.y, 0), this.z + vectors.reduce((sum, v) => sum + v.z, 0));
  }
  subtract(v) {
    return new Vector3(this.x - v.x, this.y - v.y, this.z - v.z);
  }
  dot(v) {
    return this.x * v.x + this.y * v.y + this.z * v.z;
  }
  cross(v) {
    return new Vector3(this.y * v.z - this.z * v.y, this.z * v.x - this.x * v.z, this.x * v.y - this.y * v.x);
  }
  normalize() {
    const magnitude = this.magnitude();
    return new Vector3(this.x / magnitude, this.y / magnitude, this.z / magnitude);
  }
  magnitude() {
    return Math.sqrt(this.x ** 2 + this.y ** 2 + this.z ** 2);
  }
  squaredMagnitude() {
    return this.x ** 2 + this.y ** 2 + this.z ** 2;
  }
  scale(scalar) {
    return new Vector3(this.x * scalar, this.y * scalar, this.z * scalar);
  }
  project(v) {
    const scalar = this.dot(v) / v.magnitude() ** 2;
    return v.scale(scalar);
  }
  reflect(normal) {
    return this.subtract(normal.scale(2 * this.dot(normal)));
  }
  rotateX(angle) {
    const rad = (angle * Math.PI) / 180;
    const cos = Math.cos(rad);
    const sin = Math.sin(rad);
    return new Vector3(this.x, this.y * cos - this.z * sin, this.y * sin + this.z * cos);
  }
  rotateY(angle) {
    const rad = (angle * Math.PI) / 180;
    const cos = Math.cos(rad);
    const sin = Math.sin(rad);
    return new Vector3(this.x * cos + this.z * sin, this.y, -this.x * sin + this.z * cos);
  }
  rotateZ(angle) {
    const rad = (angle * Math.PI) / 180;
    const cos = Math.cos(rad);
    const sin = Math.sin(rad);
    return new Vector3(this.x * cos - this.y * sin, this.x * sin + this.y * cos, this.z);
  }
  rotateAround(axis, angle) {
    const rad = (angle * Math.PI) / 180;
    const cos = Math.cos(rad);
    const sin = Math.sin(rad);
    const c = 1 - cos;
    const x = this.x;
    const y = this.y;
    const z = this.z;
    const u = axis.normalize();
    const ux = u.x;
    const uy = u.y;
    const uz = u.z;
    const x1 = x * (cos + ux ** 2 * c) + y * (ux * uy * c - uz * sin) + z * (ux * uz * c + uy * sin);
    const y1 = x * (uy * ux * c + uz * sin) + y * (cos + uy ** 2 * c) + z * (uy * uz * c - ux * sin);
    const z1 = x * (uz * ux * c - uy * sin) + y * (uz * uy * c + ux * sin) + z * (cos + uz ** 2 * c);
    return new Vector3(x1, y1, z1);
  }
  toArray() {
    return [this.x, this.y, this.z];
  }
  distanceTo(v) {
    return Math.sqrt((this.x - v.x) ** 2 + (this.y - v.y) ** 2 + (this.z - v.z) ** 2);
  }
  angleTo(v) {
    const dotProduct = this.dot(v);
    const magnitudeA = this.magnitude();
    const magnitudeB = v.magnitude();
    const cosTheta = dotProduct / (magnitudeA * magnitudeB);
    return Math.acos(cosTheta);
  }
  clone() {
    return new Vector3(this.x, this.y, this.z);
  }
  toString() {
    return `(${this.x}, ${this.y}, ${this.z})`;
  }
  manhattanDistanceTo(v) {
    return Math.abs(this.x - v.x) + Math.abs(this.y - v.y) + Math.abs(this.z - v.z);
  }
  manhattanLength() {
    return Math.abs(this.x) + Math.abs(this.y) + Math.abs(this.z);
  }
  static getCaretPositionFrom(position, rotation, caret) {
    const yaw = MathUtils.degreesToRadians(rotation.y);
    const pitch = MathUtils.degreesToRadians(-rotation.x);
    const pos = position ?? new Vector3(0, 0, 0);
    const cosYaw = Math.cos(yaw);
    const sinYaw = Math.sin(yaw);
    const cosPitch = Math.cos(pitch);
    const sinPitch = Math.sin(pitch);
    const forward = new Vector3(-sinYaw, 0, cosYaw);
    const left = new Vector3((cosYaw * cosPitch), sinPitch, (sinYaw * cosPitch));
    const up = new Vector3((cosYaw * -sinPitch), cosPitch, (sinYaw * -sinPitch));
    const x = pos.x + (caret.x * left.x) + (caret.y * up.x) + (caret.z * forward.x);
    const y = pos.y + (caret.x * left.y) + (caret.y * up.y) + (caret.z * forward.y);
    const z = pos.z + (caret.x * left.z) + (caret.y * up.z) + (caret.z * forward.z);
    return new Vector3(x, y, z);
  }
  getCaretPosition(rotation, caret) {
    return Vector3.getCaretPositionFrom(this, rotation, caret);
  }
}
export class MathUtils {
    static clamp(value, min, max) {
        return Math.max(min, Math.min(max, value));
    }
    static lerp(start, end, t) {
        return start + (end - start) * t;
    }
    static lerpAngle(start, end, t) {
        const angle = ((end - start + 180) % 360) - 180;
        return start + angle * t;
    }
    static degreesToRadians(degrees) {
        return degrees * (Math.PI / 180);
    }
    static radiansToDegrees(radians) {
        return radians * (180 / Math.PI);
    }
    static approximatelyEqual(a, b, tolerance = 0.00001) {
        return Math.abs(a - b) <= tolerance;
    }
    static randomInt(min, max) {
        return Math.floor(Math.random() * (max - min + 1)) + min;
    }
    static randomFloat(min, max) {
        return Math.random() * (max - min) + min;
    }
    static positionToChunk(position) {
        return {
            x: position.x >> 4,
            z: position.z >> 4,
        };
    }
    static chunkToPosition(chunk) {
        return {
            x: chunk.x << 4,
            y: 0,
            z: chunk.z << 4,
        };
    }
    static wrap(value, min, max) {
        const rangeSize = max - min + 1;
        if (value < min) {
            value += rangeSize * (Math.floor((min - value) / rangeSize) + 1);
        }
        return min + ((value - min) % rangeSize);
    }
    static wrapAngle(angle) {
        return this.wrap(angle, 0, 359);
    }
    static wrapAngle180(angle) {
        return this.wrap(angle, -180, 179);
    }
    static normalizeAngle(angle) {
        return angle - Math.floor(angle / 360) * 360;
    }
    static roundToNearestMultiple(value, multiple) {
        return Math.round(value / multiple) * multiple;
    }
}